<?php
/**
 * Plugin Name: Medic Extension
 * Description: This plugin can be used only for Medic theme.
 * Version: 1.1
 * Author: MhrTheme
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: medic-extension
 * Domain Path: /languages
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Main Elementor Extension Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.1
 */
final class Elementor_Medic_Extension {

	/**
	 * Plugin Version
	 *
	 * @since 1.1
	 *
	 * @var string The plugin version.
	 */
	const VERSION = '1.1';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.1
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */
	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.1
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '5.6';

	/**
	 * Instance
	 *
	 * @since 1.1
	 *
	 * @access private
	 * @static
	 *
	 * @var Elementor_Medic_Extension The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.1
	 *
	 * @access public
	 * @static
	 *
	 * @return Elementor_Medic_Extension An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.1
	 *
	 * @access public
	 */
	public function __construct() {

		add_action( 'init', [ $this, 'i18n' ] );
		add_action( 'plugins_loaded', [ $this, 'init' ] );
		add_action('elementor/init', [$this, 'medic_register_new_category']);

	}

	/**
	 * Load Textdomain
	 *
	 * Load plugin localization files.
	 *
	 * Fired by `init` action hook.
	 *
	 * @since 1.1
	 *
	 * @access public
	 */
	public function i18n() {

		load_plugin_textdomain( 'medic-extension' );

	}

	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.1
	 *
	 * @access public
	 */
	public function init() {

		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return;
		}

		// Add Plugin actions
		add_action( 'elementor/widgets/widgets_registered', [ $this, 'init_widgets' ] );
	}

	// Add Plugin new category
	public function medic_register_new_category() {
		\Elementor\Plugin::instance()->elements_manager->add_category(
			'medic-elements',
			[
				'title' => esc_html__('Medic Addons', 'medic-extension'),
				'icon'  => 'fa fa-image'
			]
		);
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.1
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'medic-extension' ),
			'<strong>' . esc_html__( 'Elementor Medic Extension', 'medic-extension' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'medic-extension' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.1
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'medic-extension' ),
			'<strong>' . esc_html__( 'Elementor Medic Extension', 'medic-extension' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'medic-extension' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.1
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'medic-extension' ),
			'<strong>' . esc_html__( 'Elementor Medic Extension', 'medic-extension' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'medic-extension' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.1
	 *
	 * @access public
	 */
	public function init_widgets() {

		// Include Widget files
		require_once( __DIR__ . '/el-widgets/medic-banner.php' );
		require_once( __DIR__ . '/el-widgets/medic-slider.php' );
		require_once( __DIR__ . '/el-widgets/medic-about.php' );
		require_once( __DIR__ . '/el-widgets/medic-appointment.php' );
		require_once( __DIR__ . '/el-widgets/medic-department.php' );
		require_once( __DIR__ . '/el-widgets/medic-doctor.php' );
		require_once( __DIR__ . '/el-widgets/medic-testimonial.php' );
		require_once( __DIR__ . '/el-widgets/medic-blog.php' );
		require_once( __DIR__ . '/el-widgets/medic-faq.php' );
		require_once( __DIR__ . '/el-widgets/medic-contact.php' );
		require_once( __DIR__ . '/el-widgets/medic-product.php' );

		// Register widget
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Medic_Banner_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Medic_Slider_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Medic_About_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Medic_Appointment_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Medic_Department_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Medic_Doctor_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Medic_Testimonial_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Medic_Blog_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Medic_Faq_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Medic_Contact_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Medic_Product_Widget() );
	}

}

Elementor_Medic_Extension::instance();

/**
 * Include Custom Widget.
 */
if(file_exists(dirname(__File__).'/includes/site-widgets/medic-about-widget.php')) {
	require_once(dirname(__File__).'/includes/site-widgets/medic-about-widget.php');
}

if(file_exists(dirname(__File__).'/includes/site-widgets/medic-contact-widget.php')) {
	require_once(dirname(__File__).'/includes/site-widgets/medic-contact-widget.php');
}

if(file_exists(dirname(__File__).'/includes/site-widgets/medic-open-time-widget.php')) {
	require_once(dirname(__File__).'/includes/site-widgets/medic-open-time-widget.php');
}

/**
 * Include Custom Icons.
 */
if(file_exists(dirname(__File__).'/includes/medic-extension-icons.php')) {
	require_once(dirname(__File__).'/includes/medic-extension-icons.php');
}

/**
 * Include Department Icons.
 */
if(file_exists(dirname(__File__).'/includes/medic-extension-department-icons.php')) {
	require_once(dirname(__File__).'/includes/medic-extension-department-icons.php');
}

/**
 * Include Blog Sidebar.
 */
if(file_exists(dirname(__File__).'/includes/medic-extension-blog-sidebar.php')) {
	require_once(dirname(__File__).'/includes/medic-extension-blog-sidebar.php');
}

/**
 * Include Department Sidebar.
 */
if(file_exists(dirname(__File__).'/includes/medic-extension-department-sidebar.php')) {
	require_once(dirname(__File__).'/includes/medic-extension-department-sidebar.php');
}

/**
 * Include Page Sidebar.
 */
if(file_exists(dirname(__File__).'/includes/medic-extension-page-sidebar.php')) {
	require_once(dirname(__File__).'/includes/medic-extension-page-sidebar.php');
}

// Contact Forms List
function medic_get_wpcf7_list() {
    $options = array();
    if (function_exists('wpcf7')) {
        $wpcf7_form_list = get_posts(array(
            'post_type' => 'wpcf7_contact_form',
            'showposts' => 999,
        ));
        $options[0] = esc_html('Select a Contact Form', 'medic-extension');
        if (!empty($wpcf7_form_list) && !is_wp_error($wpcf7_form_list)) {
            foreach ($wpcf7_form_list as $post) {
                $options[$post->ID] = $post->post_title;
            }
        } else {
            $options[0] = esc_html('Create a Form First', 'medic-extension');
        }
    }
    return $options;
}

/**
 * Blog Category List
 */
function medic_get_blog_cat_list() {
	$blog_category_id = get_queried_object_id();
	$medic_blog_args = array(
		'parent' => $blog_category_id
	);
	$medic_blog_terms = get_terms( 'category', get_the_ID());
	$medic_cat_options = array( '' => __('Select Category', 'medic-extension') );
	if ($medic_blog_terms) {
		foreach ($medic_blog_terms as $term) {
			$medic_cat_options[$term->name] = $term->name;
		}
	}
	return $medic_cat_options;
}

/**
 * Blog Tag List
 */
function medic_get_blog_tag_list() {
	$blog_tag_id = get_queried_object_id();
	$medic_blog_args = array(
		'parent' => $blog_tag_id
	);
	$medic_blog_terms = get_terms( 'post_tag', get_the_ID());
	$medic_tag_options = array( '' => __('Select Tag', 'medic-extension') );
	if ($medic_blog_terms) {
		foreach ($medic_blog_terms as $term) {
			$medic_tag_options[$term->name] = $term->name;
		}
	}
	return $medic_tag_options;
}

/**
 * Default Page list
 */
function medic_get_page_as_list() {
    $args = wp_parse_args(array(
        'post_type' => 'page',
        'numberposts' => -1,
    ));

    $posts = get_posts($args);
    $post_options = array(esc_html__('Select A Page', 'medic-extension') => '');

    if ($posts) {
        foreach ($posts as $post) {
            $post_options[$post->post_title] = $post->ID;
        }
    }
    $flipped = array_flip($post_options);
    return $flipped;
}

/**
 * Elementor Version Check For Icon Library
 */
function medic_is_elementor_version($operator = '<', $version = '1.1') {
	return defined('ELEMENTOR_VERSION') && version_compare(ELEMENTOR_VERSION, $version, $operator);
}

/**
 * Elementor Icon Library
 */
function medic_render_icon($settings = [], $old_icon_id = 'icon', $new_icon_id = 'selected_icon', $attributes = []) {
	$migrated = isset($settings['__fa4_migrated'][$new_icon_id]);
	$is_new = empty($settings[$old_icon_id]);

	$attributes['aria-hidden'] = 'true';

	if (medic_is_elementor_version('>=', '1.1') && ($is_new || $migrated)) {
		\Elementor\Icons_Manager::render_icon($settings[$new_icon_id], $attributes);
	} else {
		if (empty($attributes['class'])) {
			$attributes['class'] = $settings[$old_icon_id];
		} else {
			if (is_array($attributes['class'])) {
				$attributes['class'][] = $settings[$old_icon_id];
			} else {
				$attributes['class'] .= ' ' . $settings[$old_icon_id];
			}
		}
		printf('<i %s></i>', \Elementor\Utils::render_html_attributes($attributes));
	}
}

/**
 * Custom Posts Register
 */
function medic_custom_posts() {

    register_post_type('medic-departments',
        array(
            'labels' => array(
                'name' => esc_html__('Departments', 'medic-extension'),
                'singular_name' => esc_html__('Department', 'medic-extension'),
            ),
            'menu_icon'             => 'dashicons-feedback',
            'supports'              => array('title', 'editor', 'thumbnail'),
            'public'                => true,
            'show_in_rest'          => true,
            'publicly_queryable'    => true,
            'has_archive'           => false,
            'taxonomies '           => true,
            'show_ui'               => true,
            'show_in_menu'          => true,
            'query_var'             => true,
            'exclude_from_search'   => false,
            'capability_type'       => 'post',
            'yarpp_support'         => true
        )
	);
}

add_action('init', 'medic_custom_posts');

/**
 * Product Category List
 */
function medic_get_products_cat_list() {
	$products_category_id = get_queried_object_id();
	$args = array(
		'parent' => $products_category_id
	);

	$terms = get_terms( 'product_cat', get_the_ID());
	$cat_options = array(esc_html__('', 'medic-extension') => 'Select A Category');

	if ($terms) {
		foreach ($terms as $term) {
			$cat_options[$term->name] = $term->name;
		}
	}
	return $cat_options;
}
